using System;
using System.Data;
using System.Collections;
using System.Text.RegularExpressions;
using System.Text;
using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;


using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using System.Reflection;
//
namespace gov.va.med.vbecs.BOL
{
	#region Header
		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>Defines the IssuedUnit object.</summary>
    #endregion

	public class IssuedUnit : BaseBusinessObject, IRecordStatusCode
	{
        
		#region variables

		private DataTable _issuedUnitDataTable;
		//
		private DataRow _issuedUnitDataRow;

		private Common.RecordStatusCode _recordStatusCode;

		private DateTime _specimenExpirationDate;

		private Guid _patientGuid;
		
		/// <summary>
		/// Ordered Unit row version needed for release from assignment
		/// </summary>
		private byte[] _orderedUnitRowVersion;

		/// <summary>
		/// Ordered Unit Guid needed for release from assignment
		/// </summary>
		private Guid _orderedUnitGuid;

		/// <summary>
		/// Hospital locations from VistALink
		/// </summary>
		private DataTable _dtLocations;

		/// <summary>
		/// Indicates whether the current unit is being emergency issed
		/// </summary>
		private bool _emergencyIssue;

		/// <summary>
		/// Holds a collection of Hashtables.  The key for this is a BloodUnitGuid.
		/// The value is another Hashtable which holds all of the warning messages for
		/// a unit (key) and whether or not the unit can be issued (value).  
		/// 
		/// The point of this is to keep all of the message in memory so constant trips 
		/// to the DB are prevented.
		/// </summary>
		private Hashtable _unitWarnings;

		#endregion
        

		#region constructors

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4868"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>IssuedUnit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4869"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create an instance of IssuedUnit defaulting to a "new" object.
		/// </summary>
		public IssuedUnit()
		{
			GetIssuedUnitData(Guid.NewGuid());
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4932"> 
		///		<ExpectedInput>IssuedUnit Guid</ExpectedInput>
		///		<ExpectedOutput>IssuedUnit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4933"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create an instance of IssuedUnit from a Guid passed by the user with
		/// default read-only property.
		/// Specified item is retrieved from the database.
		/// </summary>
		/// <param name="issuedUnitGuid">IssuedUnit item Guid identifier</param>
		public IssuedUnit(Guid  issuedUnitGuid) 
		{
			_unitWarnings = new Hashtable();
			GetIssuedUnitData(issuedUnitGuid);
		}

		#endregion


		#region Methods
        
		
		/// <summary>
		/// Create a new instance of IssuedUnit data
		/// </summary>
		private void MakeNew()
		{
			this.IsNew = true;
			this.IsReadOnly = false;
			//
			_issuedUnitDataRow = _issuedUnitDataTable.NewRow();
			this.IssuedUnitGuid = Guid.NewGuid();
			this.LastUpdateUser = Common.LogonUser.LogonUserName;
		}

		/// <summary>
		/// Handler for dirty flag and attempts to modify a non-writable object
		/// </summary>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}
        
		/// <summary>
		/// Retrieves the IssuedUnit from the DAL
		/// </summary>
		/// <param name="issuedUnitGUID">IssuedUnit item to get Guid identifier</param>
		private void GetIssuedUnitData(System.Guid issuedUnitGUID)
		{
			_issuedUnitDataTable = DAL.IssuedUnit.GetIssuedUnit(issuedUnitGUID);
			if (_issuedUnitDataTable.Rows.Count > 0) 
			{
				_issuedUnitDataRow = _issuedUnitDataTable.Rows[0];
			}
			else 
			{
				MakeNew();
			}
			_issuedUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}


		/// <summary>
		/// GetReturnIssuedUnitDataTable
		/// </summary>
		/// <returns></returns>
		private static DataTable GetReturnIssuedUnitDataTable()
		{
			DataTable dtReturnIssuedUnits = new DataTable(TABLES.IssuedUnit.TableName);
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.IssuedUnitGuid, typeof(Guid));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.ReturnDateTime, typeof(DateTime));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.ReturnTechId, typeof(string));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.ReturnFromLocationName, typeof(string));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.ReturnSatisfactoryIndicator, typeof(bool));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.LastUpdateUser, typeof(string));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.LastUpdateFunctionId, typeof(int));
			dtReturnIssuedUnits.Columns.Add(TABLES.BloodUnit.DivisionCode, typeof(string));
			dtReturnIssuedUnits.Columns.Add(TABLES.IssuedUnit.RowVersion, typeof(byte[]));
			return dtReturnIssuedUnits;
		}


		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/10/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4694"> 
		///		<ExpectedInput>Valid patient unique identifier</ExpectedInput>
		///		<ExpectedOutput>ArrayList containing all units issued for a patient</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4695"> 
		///		<ExpectedInput>Empty patient unique identifier</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="1" testid ="3395"> 
		///		<ExpectedInput>Invalid patient unique identifier</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///	<summary>
		/// Returns all units currently issed to the supplied patient within the division
		/// Implementing BR_59.07 
		/// </summary>
		/// <param name="patientGuid">Patient unique identifier</param>
		/// <returns>Array list of issued unit information</returns>
		public static ArrayList GetIssuedUnitsForPatient(System.Guid patientGuid)
		{
			if (patientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient guid").ResString);
			}

			ArrayList units = new ArrayList();
			DataTable dtIssuedUnits = DAL.IssuedUnit.GetIssuedUnitsForPatient(patientGuid);
			foreach (DataRow dtRow in dtIssuedUnits.Rows)
			{
				Hashtable ht = new Hashtable();
				ht.Add(TABLES.IssuedUnit.BloodUnitGuid, (Guid) dtRow[TABLES.IssuedUnit.BloodUnitGuid]);
				ht.Add(TABLES.IssuedUnit.IssuedUnitGuid, (Guid) dtRow[TABLES.IssuedUnit.IssuedUnitGuid]);
				ht.Add(TABLES.BloodUnit.EyeReadableUnitId, (string) dtRow[TABLES.BloodUnit.EyeReadableUnitId]);
				ht.Add(TABLES.BloodProduct.ProductShortName, (string) dtRow[TABLES.BloodProduct.ProductShortName]);
				ht.Add(ARTIFICIAL.ABORh, (string) dtRow[ARTIFICIAL.ABORh]);
				ht.Add(TABLES.BloodUnitMedia.UnitExpirationDate, (DateTime) dtRow[TABLES.BloodUnitMedia.UnitExpirationDate]);
				if(dtRow.IsNull(TABLES.IssuedUnit.ReturnFromLocationName))
				{
					ht.Add(TABLES.IssuedUnit.ReturnFromLocationName, String.Empty);
				}
				else
				{
					ht.Add(TABLES.IssuedUnit.ReturnFromLocationName, (string) dtRow[TABLES.IssuedUnit.ReturnFromLocationName]);
				}
				ht.Add(TABLES.IssuedUnit.IssueDateTime, (DateTime) dtRow[TABLES.IssuedUnit.IssueDateTime]);
				ht.Add(TABLES.IssuedUnit.TransporterName, (string) dtRow[TABLES.IssuedUnit.TransporterName]);
				ht.Add(TABLES.IssuedUnit.IssueToLocationName, (string) dtRow[TABLES.IssuedUnit.IssueToLocationName]);
				if(dtRow.IsNull(TABLES.IssuedUnit.RemoteStorageText))
				{
					ht.Add(TABLES.IssuedUnit.RemoteStorageText, String.Empty);
				}
				else
				{
					ht.Add(TABLES.IssuedUnit.RemoteStorageText, (string) dtRow[TABLES.IssuedUnit.RemoteStorageText]);
				}
				if(dtRow.IsNull(TABLES.PatientSpecimen.SpecimenExpirationDate))
				{
					ht.Add(TABLES.PatientSpecimen.SpecimenExpirationDate, DateTime.MinValue);
				}
				else
				{
					ht.Add(TABLES.PatientSpecimen.SpecimenExpirationDate, (DateTime) dtRow[TABLES.PatientSpecimen.SpecimenExpirationDate]);
				}
				if(dtRow.IsNull(ARTIFICIAL.OrderedUnitRowVersion))
				{
					ht.Add(ARTIFICIAL.OrderedUnitRowVersion, null);
				}
				else
				{
					ht.Add(ARTIFICIAL.OrderedUnitRowVersion, (byte[]) dtRow[ARTIFICIAL.OrderedUnitRowVersion]);
				}
				if(dtRow.IsNull(TABLES.OrderedUnit.OrderedUnitGuid))
				{
					ht.Add(TABLES.OrderedUnit.OrderedUnitGuid, Guid.Empty);
				}
				else
				{
					ht.Add(TABLES.OrderedUnit.OrderedUnitGuid, (Guid) dtRow[TABLES.OrderedUnit.OrderedUnitGuid]);
				}
				ht.Add(TABLES.Patient.PatientGuid, patientGuid);

				//Adding BloodUnitStatusGuid for Locking
				ht.Add(TABLES.BloodUnitStatus.BloodUnitStatusGuid, (Guid) dtRow[TABLES.BloodUnitStatus.BloodUnitStatusGuid]);

				string fullProductCode = BOL.BloodUnit.GetFullProductCode((Common.BarcodeType)Convert.ToChar(dtRow[TABLES.BloodUnit.LabelTypeCode]),dtRow[TABLES.BloodUnit.UnitProductCode].ToString(),Convert.ToChar(dtRow[TABLES.DonationType.DonationTypeCode]),dtRow[TABLES.BloodUnit.UnitDivisionId].ToString());
				ht.Add(Common.DatabaseConstants.ArtificialColumnNames.FullProductCode, fullProductCode);

				units.Add(ht);
			}
			return units;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7475"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7476"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRemoteStorageLocations
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetRemoteStorageLocations()
		{
			return DAL.IssuedUnit.GetRemoteStorageLocations();
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4870"> 
		///		<ExpectedInput>BloodUnitGuid that has units issued for it</ExpectedInput>
		///		<ExpectedOutput>ArrayList containing issued units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4871"> 
		///		<ExpectedInput>New Guid (no issued units)</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return all units currently issed within the division
		/// Implementing BR_59.05 
		/// </summary>
		/// <param name="bloodUnitGuidsList">bloodUnitGuidsList</param>
		/// <returns>Array list of issued unit information</returns>
		public static ArrayList GetIssuedUnitsByBloodUnitGuid(ArrayList bloodUnitGuidsList)
		{
			ArrayList units = new ArrayList();
			foreach (Guid bloodUnitGuid in bloodUnitGuidsList)
			{
				DataTable dtIssuedUnits = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid(bloodUnitGuid);
				//If the unit is assigned to more than one patient I will get more rows back
				//but all of them will have the same unit info, so I only care about one of them
				Hashtable ht = new Hashtable();
				ht.Add(TABLES.IssuedUnit.BloodUnitGuid, (Guid) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.BloodUnitGuid]);
				ht.Add(TABLES.IssuedUnit.IssuedUnitGuid, (Guid) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.IssuedUnitGuid]);
				ht.Add(TABLES.BloodUnit.EyeReadableUnitId, (string) dtIssuedUnits.Rows[0][TABLES.BloodUnit.EyeReadableUnitId]);
				ht.Add(TABLES.BloodProduct.ProductShortName, (string) dtIssuedUnits.Rows[0][TABLES.BloodProduct.ProductShortName]);
				ht.Add(ARTIFICIAL.ABORh, (string) dtIssuedUnits.Rows[0][ARTIFICIAL.ABORh]);
				ht.Add(TABLES.BloodUnitMedia.UnitExpirationDate, (DateTime) dtIssuedUnits.Rows[0][TABLES.BloodUnitMedia.UnitExpirationDate]);
				if(dtIssuedUnits.Rows[0].IsNull(TABLES.IssuedUnit.ReturnFromLocationName))
				{
					ht.Add(TABLES.IssuedUnit.ReturnFromLocationName, String.Empty);
				}
				else
				{
					ht.Add(TABLES.IssuedUnit.ReturnFromLocationName, (string) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.ReturnFromLocationName]);
				}
				ht.Add(TABLES.IssuedUnit.IssueDateTime, (DateTime) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.IssueDateTime]);
				ht.Add(TABLES.IssuedUnit.TransporterName, (string) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.TransporterName]);
				ht.Add(TABLES.IssuedUnit.IssueToLocationName, (string) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.IssueToLocationName]);
				if(dtIssuedUnits.Rows[0].IsNull(TABLES.IssuedUnit.RemoteStorageText))
				{
					ht.Add(TABLES.IssuedUnit.RemoteStorageText, String.Empty);
				}
				else
				{
					ht.Add(TABLES.IssuedUnit.RemoteStorageText, (string) dtIssuedUnits.Rows[0][TABLES.IssuedUnit.RemoteStorageText]);
				}
				if(dtIssuedUnits.Rows[0].IsNull(TABLES.PatientSpecimen.SpecimenExpirationDate))
				{
					ht.Add(TABLES.PatientSpecimen.SpecimenExpirationDate, DateTime.MinValue);
				}
				else
				{
					ht.Add(TABLES.PatientSpecimen.SpecimenExpirationDate, (DateTime) dtIssuedUnits.Rows[0][TABLES.PatientSpecimen.SpecimenExpirationDate]);
				}
				if(dtIssuedUnits.Rows[0].IsNull(ARTIFICIAL.OrderedUnitRowVersion))
				{
					ht.Add(ARTIFICIAL.OrderedUnitRowVersion, null);
				}
				else
				{
					ht.Add(ARTIFICIAL.OrderedUnitRowVersion, (byte[]) dtIssuedUnits.Rows[0][ARTIFICIAL.OrderedUnitRowVersion]);
				}
				if(dtIssuedUnits.Rows[0].IsNull(TABLES.OrderedUnit.OrderedUnitGuid))
				{
					ht.Add(TABLES.OrderedUnit.OrderedUnitGuid, Guid.Empty);
				}
				else
				{
					ht.Add(TABLES.OrderedUnit.OrderedUnitGuid, (Guid) dtIssuedUnits.Rows[0][TABLES.OrderedUnit.OrderedUnitGuid]);
				}
				ht.Add(TABLES.Patient.PatientGuid, (Guid) dtIssuedUnits.Rows[0][TABLES.Patient.PatientGuid]);

				//Adding BloodUnitStatusGuid for Locking
				ht.Add(TABLES.BloodUnitStatus.BloodUnitStatusGuid, (Guid)dtIssuedUnits.Rows[0][TABLES.BloodUnitStatus.BloodUnitStatusGuid]);

				string fullProductCode = BOL.BloodUnit.GetFullProductCode((Common.BarcodeType)Convert.ToChar(dtIssuedUnits.Rows[0][TABLES.BloodUnit.LabelTypeCode]),dtIssuedUnits.Rows[0][TABLES.BloodUnit.UnitProductCode].ToString(),Convert.ToChar(dtIssuedUnits.Rows[0][TABLES.DonationType.DonationTypeCode]),dtIssuedUnits.Rows[0][TABLES.BloodUnit.UnitDivisionId].ToString());
				ht.Add(Common.DatabaseConstants.ArtificialColumnNames.FullProductCode, fullProductCode);
				units.Add(ht);
			}
			return units;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6315"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6316"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static function returns an issued unit record for a given blood unit guid
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static BOL.IssuedUnit GetIssuedUnitFromBloodUnitGuid(Guid bloodUnitGuid)
		{
			DataTable dtIssuedUnits = DAL.IssuedUnit.GetIssuedUnitByBloodUnitGuid(bloodUnitGuid);
			//If the unit is assigned to more than one patient I will get more rows back
			//but all of them will have the same unit info, so I only care about one of them
			return new BOL.IssuedUnit((Guid)dtIssuedUnits.Rows[0][TABLES.IssuedUnit.IssuedUnitGuid]);
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4872"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4873"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// Gets the tooltip messages for the listbox.
		/// </summary>
		/// <returns></returns>
		public string GetMessagesForTooltip()
		{
			StringBuilder messageString = new StringBuilder();
			
			IDictionaryEnumerator enumerator = _unitWarnings.GetEnumerator();

			while (enumerator.MoveNext()) 
			{
				if (BloodUnitGuid != (Guid) enumerator.Key) 
				{
					continue;
				}
				else
				{
					Hashtable messages = (Hashtable) enumerator.Value;
					IDictionaryEnumerator innerEnumerator = messages.GetEnumerator();
					while (innerEnumerator.MoveNext())
					{
						int divider = ((Common.StrRes.StringResourceItem) innerEnumerator.Key).ResString.IndexOf(".");
						string message = ((Common.StrRes.StringResourceItem) innerEnumerator.Key).ResString.Substring(0,divider + 1);
						messageString.Append(message + "\n");
					}
					break;
				}
			}
			return messageString.ToString();
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4874"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>WarningMessage property has value</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4875"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves and stores all of the messages.  They are stored because the messages
		/// are displayed multiple times in the applications.  Keeping them in memory will 
		/// prevent trips to the DB.
		/// Checks for conditions that would generate a warning messages for the unit
		/// BR_59.12, BR_59.13
		/// </summary>
		public void SetWarningMessages()
		{
			Hashtable messages = new Hashtable();
			//Get division current datetime
			System.DateTime _currentDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();

			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(BloodUnitGuid);

			// BR_59.12 
			// Check if Unit is expired
			if (bloodUnit.BloodUnitMedia.UnitExpirationDate < _currentDate)
			{
				messages.Add(Common.StrRes.ConfMsg.UC073.UnitExpired(), true);
			}

			// BR_59.13 
			// Check if specimen is expired
			if (this.SpecimenExpirationDate != DateTime.MinValue && this.SpecimenExpirationDate < _currentDate)
			{
				messages.Add(Common.StrRes.ConfMsg.UC073.SpecimenExpired(), true);
			}

			_unitWarnings.Add(bloodUnit.BloodUnitGuid, messages);	
	
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4771"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable containing issued units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4772"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return all units currently issed within the division
		/// </summary>
		/// <returns>Data table of issued unit information</returns>
		public static DataTable GetIssuedUnits()
		{
			return DAL.IssuedUnit.GetIssuedUnits();
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3924"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList of locations</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4934"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a list of Issue To locations based on a VistALink query
		/// BR_59.06 - Used to populate the Return from location list in PL
		/// </summary>
		/// <returns></returns>
		public ArrayList GetLocations()
		{
			ArrayList locations = new ArrayList();

			// get hospital locations within the division; BR_59.10 
			DataSet dsLocations = DAL.VAL.HospitalLocation.GetHospitalLocationsByDivision(Common.LogonUser.LogonUserDivisionCode.Trim());
			if (dsLocations.Tables.Count==1)
			{
				_dtLocations = dsLocations.Tables[0];
			}
				
			foreach (DataRow dtRow in _dtLocations.Rows)
			{
				locations.Add(dtRow[Common.DatabaseConstants.VistALinkColumnNames.LocationName].ToString());
			}
				
			return locations;
		}


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4928"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4929"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implementation of gov.va.med.vbecs.BOL.BaseBusinessObject.LoadDataRowFromThis
		/// <param name="dtRow"></param>
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.BloodUnitGuid))
			{
				dtRow[TABLES.IssuedUnit.BloodUnitGuid] = this.BloodUnitGuid;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.InPatientLocationIndicator))
			{
				dtRow[TABLES.IssuedUnit.InPatientLocationIndicator] = this.InPatientLocationIndicator;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.IssueDateTime))
			{
				dtRow[TABLES.IssuedUnit.IssueDateTime] = this.IssueDate;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.IssuedUnitGuid))
			{
				dtRow[TABLES.IssuedUnit.IssuedUnitGuid] = this.IssuedUnitGuid;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.IssueSatisfactoryIndicator))
			{
				dtRow[TABLES.IssuedUnit.IssueSatisfactoryIndicator] = this.SatisfactoryInspectionIndicator;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.IssueTechId))
			{
				dtRow[TABLES.IssuedUnit.IssueTechId] = this.IssueTechId;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.IssueToLocationName))
			{
				dtRow[TABLES.IssuedUnit.IssueToLocationName] = this.IssueToLocationName;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.LocationIen))
			{
				dtRow[TABLES.IssuedUnit.LocationIen] = this.IssueToLocationIEN;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.LastUpdateDate))
			{
				dtRow[TABLES.IssuedUnit.LastUpdateDate] = this.LastUpdateDate;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.LastUpdateUser))
			{
				dtRow[TABLES.IssuedUnit.LastUpdateUser] = this.LastUpdateUser;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.RemoteStorageIndicator))
			{
				dtRow[TABLES.IssuedUnit.RemoteStorageIndicator] = this.RemoteStorageIndicator;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.RemoteStorageText))
			{
				dtRow[TABLES.IssuedUnit.RemoteStorageText] = this.RemoteStorageText;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.PhysicianName))
			{
				dtRow[TABLES.IssuedUnit.PhysicianName] = this.RequestedBy;
			}
			if (dtRow.Table.Columns.Contains(TABLES.IssuedUnit.TransporterName))
			{
				dtRow[TABLES.IssuedUnit.TransporterName] = this.TransporterName;
			}
			if(dtRow.Table.Columns.Contains(TABLES.IssuedUnit.ReturnFromLocationName))
			{
				dtRow[TABLES.IssuedUnit.ReturnFromLocationName] = this.ReturnFromLocationName;
			}
			if(dtRow.Table.Columns.Contains(TABLES.IssuedUnit.ReturnSatisfactoryIndicator))
			{
				dtRow[TABLES.IssuedUnit.ReturnSatisfactoryIndicator] = this.ReturnSatisfactoryIndicator;
			}
			if(dtRow.Table.Columns.Contains(TABLES.IssuedUnit.PatientGuid))
			{
				dtRow[TABLES.IssuedUnit.PatientGuid] = this.PatientGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLES.IssuedUnit.ReturnTechId))
			{
				dtRow[TABLES.IssuedUnit.ReturnTechId] = this.ReturnTechId;
			}
			if(dtRow.Table.Columns.Contains(TABLES.IssuedUnit.ReturnDateTime))
			{
				dtRow[TABLES.IssuedUnit.ReturnDateTime] = this.ReturnDateTime;
			}
			if(dtRow.Table.Columns.Contains(TABLES.IssuedUnit.RowVersion))
			{
				dtRow[TABLES.IssuedUnit.RowVersion] = this.RowVersion;
			}
			return dtRow;
		}
		
		/// <summary>
		/// Implementation of gov.va.med.vbecs.BOL.BaseBusinessObject.LoadFromDataRow
		/// <param name="dtRow"></param>
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (true) 
			{
				throw new Exception("Not implemented");
			}
		}




		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4930"> 
		///		<ExpectedInput>DataTables with Save information</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4931"> 
		///		<ExpectedInput>DataTables with invalid information</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save
		/// </summary>
		/// <param name="dtIssueUnits"></param>
		/// <param name="dtReleaseUnits"></param>
		/// <param name="dtStatusUnits"></param>
		/// <param name="dtSpecimens"></param>
		/// <param name="dtIssueUnitComments"></param>
		/// <param name="dtWorkloadEvents"></param>
		/// <param name="dtIssueExceptions"></param>
		/// <param name="dtBCEMessages"></param>
		/// <returns></returns>
		public static bool Save(DataTable dtIssueUnits, 
			DataTable dtReleaseUnits, 
			DataTable dtStatusUnits, 
			DataTable dtSpecimens, 
			DataTable dtIssueUnitComments, 
			DataTable dtWorkloadEvents,
			DataTable dtIssueExceptions,
			DataTable dtBCEMessages)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			return DAL.IssuedUnit.InsertIssuedUnits(dtIssueUnits, 
				dtReleaseUnits, 
				dtStatusUnits, 
				dtSpecimens, 
				dtIssueUnitComments, 
				dtWorkloadEvents, 
				dtIssueExceptions,
				dtBCEMessages);
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4706"> 
		///		<ExpectedInput>ArrayList of information to update</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4707"> 
		///		<ExpectedInput>Arraylist with invalid RowVersions</ExpectedInput>
		///		<ExpectedOutput>Common.RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Return issued unit update method  
		/// </summary>
		/// <param name="returnUnits"></param>
		/// <param name="returnUnitExceptions"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="patientGuid"></param>
		/// <returns>
		///     A bool value...
		/// </returns>
		public static bool Update(ArrayList returnUnits, 
			ArrayList returnUnitExceptions, 
			Common.WorkloadProcessID workloadProcessId,
			Guid patientGuid)
		{
			DataSet dsSave = new DataSet();

			// Create DataTable for return units
			DataTable dtReturnIssuedUnits = GetReturnIssuedUnitDataTable();			

			//Guid array needed for workload generation
			Guid [] bloodUnitGuids = new Guid[returnUnits.Count];
			int nCtr = 0;
			foreach (BOL.IssuedUnit issuedUnit in returnUnits)
			{
				//CR 2279
				bloodUnitGuids[nCtr] = (Guid) issuedUnit.BloodUnitGuid;
				//Add return unit info to the table 
				DataRow drReturnIssuedUnits = dtReturnIssuedUnits.NewRow();
				drReturnIssuedUnits[TABLES.IssuedUnit.IssuedUnitGuid] = (Guid) issuedUnit.IssuedUnitGuid;
				drReturnIssuedUnits[TABLES.IssuedUnit.ReturnDateTime] = (DateTime) issuedUnit.ReturnDateTime;
				drReturnIssuedUnits[TABLES.IssuedUnit.ReturnTechId] = issuedUnit.ReturnTechId;
				drReturnIssuedUnits[TABLES.IssuedUnit.ReturnFromLocationName]= issuedUnit.ReturnFromLocationName;
				// Always true:  Unsatisfactory units are saved when the checkbox is checked, not here.
				drReturnIssuedUnits[TABLES.IssuedUnit.ReturnSatisfactoryIndicator] = true;
				drReturnIssuedUnits[TABLES.IssuedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drReturnIssuedUnits[TABLES.IssuedUnit.LastUpdateFunctionId] = Common.UpdateFunction.UC059FrmReturnIssuedUnits;
				drReturnIssuedUnits[TABLES.IssuedUnit.RowVersion] = (byte[]) issuedUnit.RowVersion;
				//BR_59.10 - unit is retured within the same division
				drReturnIssuedUnits[TABLES.BloodUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				dtReturnIssuedUnits.Rows.Add(drReturnIssuedUnits);

				nCtr++;
			}

			DataTable dtReturnUnitExceptions = null;
			foreach(BOL.ExceptionReport exRep in returnUnitExceptions)
			{
				if (dtReturnUnitExceptions == null)
					dtReturnUnitExceptions = exRep.ExceptionData.Table.Clone();

				dtReturnUnitExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
			}
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			
			//Add tables to dsSave dataset
			dsSave.Tables.Add(dtReturnIssuedUnits);
			dsSave.Tables.Add(dtWorkloadEvent);

			if(dtReturnUnitExceptions != null)
				dsSave.Tables.Add(dtReturnUnitExceptions);

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit is returned from issue, send BCE message

			Guid assignedPatientGuid = Guid.Empty;
			for (int i=0; i<bloodUnitGuids.Length; i++)
			{
				DataTable dtBCEMessages = null;
				DataTable dtTempBCEMessages = null;

				//getting all patients assigned to a unit
				DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(bloodUnitGuids[i]);

				for (int j=0; j< dtPatients.Rows.Count; j++)
				{
					//CR 3092 removed the check for previous unit status
					assignedPatientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];

					//CR 3015
					OrderedUnit orderedUnit = new OrderedUnit(bloodUnitGuids[i], assignedPatientGuid);
					if (dtBCEMessages == null)
					{
						dtBCEMessages = OrderedUnit.UpdateBce(bloodUnitGuids[i], assignedPatientGuid, orderedUnit.OrderedComponentGuid, "", Common.HL7BPDispenseStatus.RS, false);
					}
					else
					{
						dtTempBCEMessages = OrderedUnit.UpdateBce(bloodUnitGuids[i], assignedPatientGuid, orderedUnit.OrderedComponentGuid, "", Common.HL7BPDispenseStatus.RS, false);
						if (dtTempBCEMessages.Rows.Count > 0)
							dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
					}
				}

				if (dtBCEMessages != null)
				{
					if (dsSave.Tables.IndexOf(dtBCEMessages.TableName) == -1)
					{
						dsSave.Tables.Add(dtBCEMessages);
					}
					else
					{
						for (int k=0 ; k < dtBCEMessages.Rows.Count; k++)
						{
							dsSave.Tables[dtBCEMessages.TableName].Rows.Add(dtBCEMessages.Rows[k].ItemArray);
						}
					}
				}
				//BCE_MESSAGE end

			}

			//Call DAL
			bool result = DAL.IssuedUnit.Update(dsSave, Common.UpdateFunction.UC059FrmReturnIssuedUnits);

			return result;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7477"> 
		///		<ExpectedInput>Valid issued unit</ExpectedInput>
		///		<ExpectedOutput>Unit marked unsatisfactory</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7478"> 
		///		<ExpectedInput>Invalid issued unit</ExpectedInput>
		///		<ExpectedOutput>Null reference exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MarkUnitUnsatisfactory
		/// BR_73.19
		/// </summary>
		/// <returns></returns>
		public static bool MarkUnitUnsatisfactory(BOL.IssuedUnit issuedUnit, 
												ArrayList returnUnitExceptions, 
												Common.WorkloadProcessID workloadProcessId)
		{
			DataTable dtReturnIssuedUnits = BOL.IssuedUnit.GetReturnIssuedUnitDataTable();			

			//Add return unit info to the table 
			DataRow drReturnIssuedUnits = dtReturnIssuedUnits.NewRow();
			drReturnIssuedUnits[TABLES.IssuedUnit.IssuedUnitGuid] = (Guid) issuedUnit.IssuedUnitGuid;
			drReturnIssuedUnits[TABLES.IssuedUnit.ReturnDateTime] = (DateTime) issuedUnit.ReturnDateTime;
			drReturnIssuedUnits[TABLES.IssuedUnit.ReturnTechId] = issuedUnit.ReturnTechId;
			drReturnIssuedUnits[TABLES.IssuedUnit.ReturnFromLocationName]= issuedUnit.ReturnFromLocationName;
			drReturnIssuedUnits[TABLES.IssuedUnit.ReturnSatisfactoryIndicator] = false;
			drReturnIssuedUnits[TABLES.IssuedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drReturnIssuedUnits[TABLES.IssuedUnit.LastUpdateFunctionId] = Common.UpdateFunction.UC059FrmReturnIssuedUnits;
			drReturnIssuedUnits[TABLES.IssuedUnit.RowVersion] = (byte[]) issuedUnit.RowVersion;
			//BR_59.10 - unit is retured within the same division
			drReturnIssuedUnits[TABLES.BloodUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtReturnIssuedUnits.Rows.Add(drReturnIssuedUnits);


			// Unit Status: Create DataTable for blood unit status
			DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);

			//Unit marked as unsatisfactory for return, the system quarantines the unit per BR_59.08
			
			//Add blood unit status info to the table
			//Create an instance of the blood unit object, so I can get the rowversion for update the record
			BloodUnitStatus tempBloodUnitStatus = new BloodUnitStatus((Guid) issuedUnit.BloodUnitGuid);
			DataRow drBloodUnitStatus = tempBloodUnitStatus.LoadDataRowFromThis(dtBloodUnitStatus.NewRow());
			drBloodUnitStatus[TABLES.BloodUnitStatus.QuarantineIndicator] = true;
			dtBloodUnitStatus.Rows.Add(drBloodUnitStatus);
	

			// Ordered Unit: If unit is marked as unsatisfactory the system releases the unit 
			// from assignment.
			DataTable dtOrderedUnit = new DataTable(TABLES.OrderedUnit.TableName);

			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.OrderedUnitGuid, typeof(Guid));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.ReleaseTechId, typeof(string));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.ReleaseDate, typeof(DateTime));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.LastUpdateUser, typeof(string));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.RowVersion, typeof(byte[]));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.LastUpdateFunctionId, typeof(int));
			dtOrderedUnit.Columns.Add(TABLES.OrderedUnit.DivisionCode, typeof(string));

			DataRow drOrderedUnit = dtOrderedUnit.NewRow();
			drOrderedUnit[TABLES.OrderedUnit.OrderedUnitGuid] = (Guid) issuedUnit.OrderedUnitGuid;
			drOrderedUnit[TABLES.OrderedUnit.ReleaseTechId] = issuedUnit.ReturnTechId;
			drOrderedUnit[TABLES.OrderedUnit.ReleaseDate] = (DateTime) issuedUnit.ReturnDateTime;
			drOrderedUnit[TABLES.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drOrderedUnit[TABLES.OrderedUnit.RowVersion] = (byte[]) issuedUnit.OrderedUnitRowVersion;
			drOrderedUnit[TABLES.OrderedUnit.LastUpdateFunctionId] = Common.UpdateFunction.UC059FrmReturnIssuedUnits;
			drOrderedUnit[TABLES.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtOrderedUnit.Rows.Add(drOrderedUnit);		


			// Exceptions
			DataTable dtReturnUnitExceptions = IssuedUnit.GetExceptions(returnUnitExceptions);


			// Workload
			Guid [] bloodUnitGuids = {issuedUnit.IssuedUnitGuid};
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			

			DataTable dtBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit is returned from issue as unsatisfactory, send BCE message

			//CR 3015
			dtBCEMessages = OrderedUnit.UpdateBceForRelease(issuedUnit.BloodUnitGuid, issuedUnit.PatientGuid, false);

			//BCE_MESSAGE end
			bool result = DAL.IssuedUnit.MarkUnitUnsatisfactory(dtReturnIssuedUnits, dtBloodUnitStatus, dtOrderedUnit, dtReturnUnitExceptions, dtWorkloadEvent, dtBCEMessages);

			return result;
		}

		private static DataTable GetExceptions(ArrayList returnUnitExceptions)
		{
			DataTable dtReturnUnitExceptions = null;
			foreach(BOL.ExceptionReport exRep in returnUnitExceptions)
			{
				if (dtReturnUnitExceptions == null)
					dtReturnUnitExceptions = exRep.ExceptionData.Table.Clone();

				dtReturnUnitExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
			}
			return dtReturnUnitExceptions;
		}

		
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4926"> 
		///		<ExpectedInput>Valid arguments</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4927"> 
		///		<ExpectedInput>Invalid orderedUnitGuid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MarkUnitUnsatisfactory
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="rowVersion"></param>
		/// <param name="unitStatusRowVersion"></param>
		/// <param name="updateFunction"></param>
		/// <param name="dtBCEMessages"></param>
		/// <returns></returns>
		public static bool MarkUnitUnsatisfactory(Guid orderedUnitGuid, byte [] rowVersion, byte [] unitStatusRowVersion, Common.UpdateFunction updateFunction, DataTable dtBCEMessages)
		{
			if (orderedUnitGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ordered unit guid").ResString);
			}

			return DAL.IssuedUnit.MarkUnitUnsatisfactory(orderedUnitGuid, rowVersion, unitStatusRowVersion, updateFunction, dtBCEMessages);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6317"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6318"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of issue-to locations
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetIssueToLocationList()
		{
			DataTable dtIssueToLocations = DAL.IssuedUnit.GetIssueToLocations();
			ArrayList issueToLocationList = new ArrayList();
			
			foreach(DataRow drIssueToLocation in dtIssueToLocations.Rows)
			{
				issueToLocationList.Add(drIssueToLocation[TABLES.IssuedUnit.IssueToLocationName].ToString());
			}

			return issueToLocationList;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6319"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6320"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of physicians
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetPhysiciansList()
		{
			DataTable dtPhysicians = DAL.IssuedUnit.GetPhysicianList();
			ArrayList physiciansList = new ArrayList();
			
			foreach(DataRow drPhysicians in dtPhysicians.Rows)
			{
				physiciansList.Add(drPhysicians[TABLES.IssuedUnit.PhysicianName].ToString());
			}

			return physiciansList;
		}

		#endregion


		#region Public Properties
       
 		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4876"> 
		///		<ExpectedInput>Issued Unit Guid</ExpectedInput>
		///		<ExpectedOutput>Issued Unit Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4877"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Guid value containing the Issued Unit Guid
		/// </summary>
		public Guid IssuedUnitGuid
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.IssuedUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (Guid)_issuedUnitDataRow[TABLES.IssuedUnit.IssuedUnitGuid];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.IssuedUnitGuid] = value;
			}
		}
       	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4878"> 
		///		<ExpectedInput>Blood Unit guid</ExpectedInput>
		///		<ExpectedOutput>Blood Unit guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4879"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Guid value containing the BloodUnit Guid
		/// </summary>
		public Guid BloodUnitGuid
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.BloodUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (Guid)_issuedUnitDataRow[TABLES.IssuedUnit.BloodUnitGuid];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.BloodUnitGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4880"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4881"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Requesting physician (optional)
		/// </summary>
		public string RequestedBy
		{
			get
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.PhysicianName))
				{
					return string.Empty;
				}
				return (string) _issuedUnitDataRow[TABLES.IssuedUnit.PhysicianName];
			}
			set
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.PhysicianName] = value;
			}
		}
      	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4882"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>Valid Date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4883"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DateTime value containing the Issue Date
		/// </summary>
		public DateTime IssueDate
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.IssueDateTime))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_issuedUnitDataRow[TABLES.IssuedUnit.IssueDateTime];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.IssueDateTime] = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4884"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4885"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// String representing the tech ID
		/// </summary>
		public string IssueTechId
		{
			get
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.IssueTechId))
				{
					return string.Empty;
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.IssueTechId];
			}
			set
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.IssueTechId] = value;
			}
		}
    	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4886"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4887"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// string value containing the Transporter Name
		/// </summary>
		public string TransporterName
		{
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.TransporterName))
				{
					return "";
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.TransporterName];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.TransporterName] = value;
			}
		}
      	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4888"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4889"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// String value containing the Issue-To-Location Name
		/// </summary>
		public string IssueToLocationName
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.IssueToLocationName))
				{
					return "";
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.IssueToLocationName];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.IssueToLocationName] = value;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4890"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4891"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// bool value containing the Remote Storage Indicator
		/// </summary>
		public bool RemoteStorageIndicator
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.RemoteStorageIndicator))
				{
					return false;
				}
				return (bool)_issuedUnitDataRow[TABLES.IssuedUnit.RemoteStorageIndicator];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.RemoteStorageIndicator] = value;
			}
		}
		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4892"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4893"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// string value containing the Remote Storage Text
		/// </summary>
		public string RemoteStorageText
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.RemoteStorageText))
				{
					return "";
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.RemoteStorageText];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.RemoteStorageText] = value;
			}
		}
        
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4894"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4895"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// bool value containing the InPatient Location Indicator
		/// </summary>
		public bool InPatientLocationIndicator
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.InPatientLocationIndicator))
				{
					return false;
				}
				return (bool)_issuedUnitDataRow[TABLES.IssuedUnit.InPatientLocationIndicator];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.InPatientLocationIndicator] = value;
			}
		}
        
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4896"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4897"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// bool value containing the Satisfactory Inspection Indicator
		/// </summary>
		public bool SatisfactoryInspectionIndicator
		{	
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.IssueSatisfactoryIndicator))
				{
					return false;
				}
				return (bool)_issuedUnitDataRow[TABLES.IssuedUnit.IssueSatisfactoryIndicator];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.IssueSatisfactoryIndicator] = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4898"> 
		///		<ExpectedInput>Valid record status code</ExpectedInput>
		///		<ExpectedOutput>Valid record status code</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4899"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4900"> 
		///		<ExpectedInput>Valid integer</ExpectedInput>
		///		<ExpectedOutput>Valid integer</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4901"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Location's Identification Entry Number
		/// </summary>
		public int IssueToLocationIEN
		{
			get
			{
				return System.Convert.ToInt32( _issuedUnitDataRow[TABLES.IssuedUnit.LocationIen] );
			}
			set
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.LocationIen] = value;
			}
		}
        		
        ///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4902"> 
		///		<ExpectedInput>Valid datetime</ExpectedInput>
		///		<ExpectedOutput>Valid datetime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4903"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// DateTime value containing the  Last Update Date
		/// </summary>
		public new DateTime LastUpdateDate
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.LastUpdateDate))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_issuedUnitDataRow[TABLES.IssuedUnit.LastUpdateDate];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.LastUpdateDate] = value;
			}
		}
       	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4904"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4905"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// string value containing the  Last Update User
		/// </summary>
		public new string LastUpdateUser
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.LastUpdateUser))
				{
					return "";
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.LastUpdateUser];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.LastUpdateUser] = value;
			}
		}
      	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/25/03</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4906"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>byte[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4907"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// byte[] value containing the  Row Version
		/// </summary>
		public new byte[] RowVersion
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.RowVersion))
				{
					return null;
				}
				return (byte[])_issuedUnitDataRow[TABLES.IssuedUnit.RowVersion];
			}
		}
   
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/14/04</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4908"> 
		///		<ExpectedInput>Bool value</ExpectedInput>
		///		<ExpectedOutput>Bool value</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4909"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// bool value containing the Return Satisfactory Indicator
		/// </summary>
		public bool ReturnSatisfactoryIndicator
		{	
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.ReturnSatisfactoryIndicator))
				{
					return true;
				}
				return (bool)_issuedUnitDataRow[TABLES.IssuedUnit.ReturnSatisfactoryIndicator];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.ReturnSatisfactoryIndicator] = value;
			}
		}
	
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/14/04</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4910"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4911"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// string value containing the Return-From-Location Name
		/// </summary>
		public string ReturnFromLocationName
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.ReturnFromLocationName))
				{
					return String.Empty;
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.ReturnFromLocationName];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.ReturnFromLocationName] = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4912"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4913"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// String representing the return tech ID
		/// </summary>
		public string ReturnTechId
		{
			get
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.ReturnTechId))
				{
					return string.Empty;
				}
				return (string)_issuedUnitDataRow[TABLES.IssuedUnit.ReturnTechId];
			}
			set
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.ReturnTechId] = value;
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/15/04</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4914"> 
		///		<ExpectedInput>Valid datetime</ExpectedInput>
		///		<ExpectedOutput>Valid datetime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4915"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Unit return date and time
		/// </summary>
		public DateTime ReturnDateTime
		{
			
			get 
			{
				if (_issuedUnitDataRow.IsNull(TABLES.IssuedUnit.ReturnDateTime))
				{
					return System.DateTime.MinValue;
				}
				return (DateTime)_issuedUnitDataRow[TABLES.IssuedUnit.ReturnDateTime];
			}
			set 
			{
				_issuedUnitDataRow[TABLES.IssuedUnit.ReturnDateTime] = value;
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/15/04</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4916"> 
		///		<ExpectedInput>Valid datetime</ExpectedInput>
		///		<ExpectedOutput>Valid datetime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4917"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Specimen Expiration date and time. Needed for UC 59.
		/// </summary>
		public DateTime SpecimenExpirationDate
		{
		
			get 
			{
				return _specimenExpirationDate;
			}
			set 
			{
				_specimenExpirationDate = value;
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/15/04</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4918"> 
		///		<ExpectedInput>Valid row version</ExpectedInput>
		///		<ExpectedOutput>Valid row version</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4919"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Orderd Unit Row Version. Needed for UC 59 when coming from Patient menu.
		/// Since I already have that info available, it's easier to assing a value to a property,
		/// than make another stored proc call. There should be only one Ordered Unit per Issued unit,
		/// that hasn't been released yet (per Carl J.)
		/// </summary>
		public byte[] OrderedUnitRowVersion
		{
			get
			{
				return _orderedUnitRowVersion;
			}
			set
			{
				_orderedUnitRowVersion = value;
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/28/04</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4920"> 
		///		<ExpectedInput>Valid guid</ExpectedInput>
		///		<ExpectedOutput>Valid guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4921"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Orderd Unit Guid. Needed for UC 59 when coming from Patient menu.
		/// Since I already have that info available, it's easier to assing a value to a property,
		/// than make another stored proc call. There should be only one Ordered Unit per Issued unit,
		/// that hasn't been released yet (per Carl J.)
		/// </summary>
		public Guid OrderedUnitGuid
		{
			get
			{
				return _orderedUnitGuid;
			}
			set
			{
				_orderedUnitGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4922"> 
		///		<ExpectedInput>Valid guid</ExpectedInput>
		///		<ExpectedOutput>Valid guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4923"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Guid value containing the Patient Guid
		/// </summary>
		public Guid PatientGuid
		{
			
			get 
			{
				return _patientGuid;
			}
			set 
			{
				_patientGuid = value;
			}
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4924"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4925"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if current unit has been out of controlled storage for more than 30 minutes
		/// If yes, exception report will be generated once the override comment is entered  
		/// BR_59.04
		/// </summary>
		public bool OutOfUncontrolledStorage
		{
			get
			{
				if(this.RemoteStorageIndicator == false && 
					(this.ReturnDateTime > this.IssueDate.AddMinutes(30)))
				{
					return true;
				}
				else
				{
					return false;
				}
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7301"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7302"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// EmergencyIssue
		/// </summary>
		public bool EmergencyIssue
		{
			get
			{
				return _emergencyIssue;
			}
			set
			{
				_emergencyIssue = value;
			}
		}

		#endregion

	}
}
